VERSION 1.0 CLASS
BEGIN
  MultiUse = -1  'True
END
Attribute VB_Name = "clsCCRegionals"
Attribute VB_GlobalNameSpace = False
Attribute VB_Creatable = False
Attribute VB_PredeclaredId = False
Attribute VB_Exposed = False
Option Compare Database
Option Explicit

'---------------------------------------------------------------------------------------
' Module     : clsCCRegionals
' Version    : 1.1
' Author     : Christian Coppes
' Date       : 23.01.2013
' Last Change: 29.05.2013
' Purpose    : Gets the regional settings of Windows and supports with convert functions
'---------------------------------------------------------------------------------------

'#######################################################################################
'---------------------------------------------------------------------------------------
'----------------------------------- Constants  ----------------------------------------
'---------------------------------------------------------------------------------------
'#######################################################################################

Private Const cMODULENAME As String = "clsCCRegionals"

Private Const VER_PLATFORM_WIN32s        As Long = 0&
Private Const VER_PLATFORM_WIN32_WINDOWS As Long = 1&
Private Const VER_PLATFORM_WIN32_NT      As Long = 2&

Private Type typWinVersionInfo
  lngWinVersionInfoSize As Long   ' Size of the structure
  lngMajorVersion       As Long
  lngMinorVersion       As Long
  lngBuildNumber        As Long
  lngPlatformId         As Long
  strSPVersion          As String * 128 ' Service Pack
End Type

' -----------------------------------------------------------
' Formats for Format-Function
Private Const cFORMAT_StandardDateTime      As String = "yyyy\/mm\/dd hh\:mm\:ss"
Private Const cFORMAT_SQLDateOnly           As String = "yyyymmdd"
Private Const cFORMAT_SQLTimeOnly           As String = "HH\:mm\:ss"
Private Const cFORMAT_SQLDateAndTime        As String = cFORMAT_SQLDateOnly & " " & cFORMAT_SQLTimeOnly

Public Enum EnmCCRegionalsTarget
    enmCCRegTarget_Access
    enmCCRegTarget_SQLServer
End Enum

Public Enum EnmCCRegionalsDateTime
    enmCCReg_DateOnly
    enmCCReg_TimeOnly
    enmCCReg_DateAndTime
End Enum

Public Enum EnmCCRegionalsDateFormat
    enmCCRegDateFormat_Short
    enmCCRegDateFormat_Long
    enmCCRegDateFormat_YearMonth
End Enum

Public Enum EnmCCRegionalsShortOrLong
    LOCALENAME_Long
    LOCALENAME_short
    LOCALENAME_Shortest
End Enum

Public Enum EnmCCRegionalsCountryOutputType
    enmCCRegOutputType_ISO3166
    enmCCRegOutputType_ibm
    enmCCRegOutputType_GeoID
    enmCCRegOutputType_LocalNameLong
    enmCCRegOutputType_NativeNameLong
    enmCCRegOutputType_EnglishNamesLong
    enmCCRegOutputType_AbbreviatedName
'    enmCCRegOutputType_DefaultLocaleNames
    enmCCRegOutputType_Index
    enmCCRegOutputType_Currency
    enmCCRegOutputType_CurrencyISO
    enmCCRegOutputType_LocaleID
End Enum

Public Enum EnmCCRegionalsWinVersion
    Version_WinXP
    Version_WinVista
    Version_Win7
End Enum

Public Enum EnmCCRegionalsType
    ' Prefix "A_" to show them as first entries in the IntelliSense list
    A_LOCALE_SYSTEM_DEFAULT& = &H800              ' Returns the locale settings of the system independent of the user
    A_LOCALE_USER_DEFAULT& = &H400                ' Returns the current locale settings of the user
    A_LOCALE_INVARIANT& = &H7F                    ' Returns standard locale settings always equal on all systems
    ' Country Codes
    LOCALE_Afrikaans& = &H436
    LOCALE_Albanian& = &H41C
    LOCALE_Arabic_UnitedArabEmirates& = &H3801
    LOCALE_Arabic_Bahrain& = &H3C01
    LOCALE_Arabic_Algeria& = &H1401
    LOCALE_Arabic_Egypt& = &HC01
    LOCALE_Arabic_Iraq& = &H801
    LOCALE_Arabic_Jordan& = &H2C01
    LOCALE_Arabic_Kuwait& = &H3401
    LOCALE_Arabic_Lebanon& = &H3001
    LOCALE_Arabic_Libya& = &H1001
    LOCALE_Arabic_Morocco& = &H1801
    LOCALE_Arabic_Oman& = &H2001
    LOCALE_Arabic_Qatar& = &H4001
    LOCALE_Arabic_SaudiArabia& = &H401
    LOCALE_Arabic_Syria& = &H2801
    LOCALE_Arabic_Tunisia& = &H1C01
    LOCALE_Arabic_Yemen& = &H2401
    LOCALE_Armenian& = &H42B
    LOCALE_Azeri_Latin& = &H42C
    LOCALE_Azeri_Cyrillic& = &H82C
    LOCALE_Basque& = &H42D
    LOCALE_Belarusian& = &H423
    LOCALE_Bulgarian& = &H402
    LOCALE_Catalan& = &H403
    LOCALE_Chinese_China& = &H804
    LOCALE_Chinese_HongKongSAR& = &HC04
    LOCALE_Chinese_MacauSAR& = &H1404
    LOCALE_Chinese_Singapore& = &H1004
    LOCALE_Chinese_Taiwan& = &H404
    LOCALE_Croatian& = &H41A
    LOCALE_Czech& = &H405
    LOCALE_Danish& = &H406
    LOCALE_Dutch_Netherlands& = &H413
    LOCALE_Dutch_Belgium& = &H813
    LOCALE_English_Australia& = &HC09
    LOCALE_English_Belize& = &H2809
    LOCALE_English_Canada& = &H1009
    LOCALE_English_Caribbean& = &H2409
    LOCALE_English_India& = &H4009
    LOCALE_English_Ireland& = &H1809
    LOCALE_English_Jamaica& = &H2009
    LOCALE_English_Malaysia& = &H4409
    LOCALE_English_NewZealand& = &H1409
    LOCALE_English_Phillippines& = &H3409
    LOCALE_English_Singapore& = &H4809
    LOCALE_English_SouthernAfrica& = &H1C09
    LOCALE_English_Trinidad& = &H2C09
    LOCALE_English_GreatBritain& = &H809
    LOCALE_English_UnitedStates& = &H409
    LOCALE_English_Zimbabwe& = &H3009
    LOCALE_Estonian& = &H425
    LOCALE_Farsi& = &H429
    LOCALE_Finnish& = &H40B
    LOCALE_Faroese& = &H438
    LOCALE_French_France& = &H40C
    LOCALE_French_Belgium& = &H80C
    LOCALE_French_Canada& = &HC0C
    LOCALE_French_Luxembourg& = &H140C
    LOCALE_French_Switzerland& = &H100C
    LOCALE_Irish_Ireland& = &H83C
    LOCALE_ScottishGaelic_UnitedKingdom& = &H43C
    LOCALE_German_Germany& = &H407
    LOCALE_German_Austria& = &HC07
    LOCALE_German_Liechtenstein& = &H1407
    LOCALE_German_Luxembourg& = &H1007
    LOCALE_German_Switzerland& = &H807
    LOCALE_Greek& = &H408
    LOCALE_Hebrew& = &H40D
    LOCALE_Hindi& = &H439
    LOCALE_Hungarian& = &H40E
    LOCALE_Icelandic& = &H40F
    LOCALE_Indonesian& = &H421
    LOCALE_Italian_Italy& = &H410
    LOCALE_Italian_Switzerland& = &H810
    LOCALE_Japanese& = &H411
    LOCALE_Korean& = &H412
    LOCALE_Latvian& = &H426
    LOCALE_Lithuanian& = &H427
    LOCALE_FYRO_Macedonia& = &H42F
    LOCALE_Malay_Malaysia& = &H43E
    LOCALE_Malay_Brunei& = &H83E
    LOCALE_Maltese& = &H43A
    LOCALE_Marathi& = &H44E
    LOCALE_Norwegian_Bokmal& = &H414
    LOCALE_Norwegian_Nynorsk& = &H814
    LOCALE_Polish& = &H415
    LOCALE_Portuguese_Portugal& = &H816
    LOCALE_Portuguese_Brazil& = &H416
    LOCALE_Raeto_Romance& = &H417
    LOCALE_Romanian_Romania& = &H418
    LOCALE_Romanian_RepublicOfMoldova& = &H818
    LOCALE_Russian& = &H419
    LOCALE_Russian_RepublicOfMoldova& = &H819
    LOCALE_Sanskrit& = &H44F
    LOCALE_Serbian_Cyrillic& = &HC1A
    LOCALE_Serbian_Latin& = &H81A
    LOCALE_Setsuana& = &H432
    LOCALE_Slovenian& = &H424
    LOCALE_Slovak& = &H41B
    LOCALE_Sorbian& = &H42E
    LOCALE_Spanish_Spain_Modern& = &HC0A
    LOCALE_Spanish_Spain_Traditional& = &H40A
    LOCALE_Spanish_Argentina& = &H2C0A
    LOCALE_Spanish_Bolivia& = &H400A
    LOCALE_Spanish_Chile& = &H340A
    LOCALE_Spanish_Colombia& = &H240A
    LOCALE_Spanish_CostaRica& = &H140A
    LOCALE_Spanish_DominicanRepublic& = &H1C0A
    LOCALE_Spanish_Ecuador& = &H300A
    LOCALE_Spanish_Guatemala& = &H100A
    LOCALE_Spanish_Honduras& = &H480A
    LOCALE_Spanish_Mexico& = &H80A
    LOCALE_Spanish_Nicaragua& = &H4C0A
    LOCALE_Spanish_Panama& = &H180A
    LOCALE_Spanish_Peru& = &H280A
    LOCALE_Spanish_PuertoRico& = &H500A
    LOCALE_Spanish_Paraguay& = &H3C0A
    LOCALE_Spanish_ElSalvador& = &H440A
    LOCALE_Spanish_Uruguay& = &H380A
    LOCALE_Spanish_Venezuela& = &H200A
    LOCALE_SouthernSotho& = &H430
    LOCALE_Swahili& = &H441
    LOCALE_Swedish_Sweden& = &H41D
    LOCALE_Swedish_Finland& = &H81D
    LOCALE_Tamil& = &H449
    LOCALE_Tatar& = &H444
    LOCALE_Thai& = &H41E
    LOCALE_Turkish& = &H41F
    LOCALE_Tsonga& = &H431
    LOCALE_Ukrainian& = &H422
    LOCALE_Urdu& = &H420
    LOCALE_Uzbek_Cyrillic& = &H843
    LOCALE_Uzbek_Latin& = &H443
    LOCALE_Vietnamese& = &H42A
    LOCALE_Xhosa& = &H434
    LOCALE_Yiddish& = &H43D
    LOCALE_Zulu& = &H435
End Enum

' W7_ and WV_ added to constant names for easier selection
Public Enum EnmCCRegionals
    LOCALE_ILANGUAGE = &H1                    '  language id (i.e. 1033 = US)
    LOCALE_SLANGUAGE = &H2                    '  English name of language, i.e. "English"
    LOCALE_SENGLANGUAGE = &H1001              '  English name of language with country, i.e. "English (United States)"
    LOCALE_SABBREVLANGNAME = &H3              '  abbreviated language name, i.e. "ENU"
    LOCALE_SNATIVELANGNAME = &H4              '  native name of language, i.e. "English"
    W7_LOCALE_SNATIVELANGUAGENAME = &H4       '  >=Win7: Replacement constant name for LOCALE_SNATIVELANGNAME
    W7_LOCALE_SLOCALIZEDLANGUAGENAME = &H6F   '  >=Win7: Full localized primary name of the user interface language included in a localized display name, for example, Deutsch representing German
    LOCALE_SISO639LANGNAME = &H59             '  ISO language name, i.e. "de" or "en" (max. 9 characters)
    WV_LOCALE_SISO639LANGNAME2 = &H67         '  >=WinVista: ISO language name, i.e. "deu" or "eng" (max. 9 characters)

    LOCALE_ICOUNTRY = &H5                     '  Country/region code, based on international phone codes, also referred to as IBM country/region codes
    LOCALE_SCOUNTRY = &H6                     '  localized name of country, deprecated for Windows 7 and later, i.e. "United States"
    W7_LOCALE_SLOCALIZEDCOUNTRYNAME = &H6     '  >=Win7: Full localized name of the country/region, for example, "Deutschland" for Germany
    LOCALE_SENGCOUNTRY = &H1002               '  English name of country, i.e. "United States"
    LOCALE_SABBREVCTRYNAME = &H7              '  abbreviated country name, i.e. "USA"
    LOCALE_SNATIVECTRYNAME = &H8              '  native name of country, i.e. "Deutschland" for Germany
    W7_LOCALE_SNATIVECOUNTRYNAME = &H8        '  >=Win7: Replacement constant name for LOCALE_SNATIVECTRYNAME
    LOCALE_IGEOID = &H5B                      '  32-bit signed number that uniquely identifies a geographical location, i.e. 244 for USA (table see http://msdn.microsoft.com/en-us/library/windows/desktop/dd374073%28v=vs.85%29.aspx)
    LOCALE_SISO3166CTRYNAME = &H5A            '  ISO country name, i.e. "DE" or "US" (max. 9 characters)
    WV_LOCALE_SISO3166CTRYNAME2 = &H68        '  >=WinVista: ISO country name, i.e. "DEU" or "USA" (max. 9 characters)

    LOCALE_IDEFAULTLANGUAGE = &H9             '  default language id, obsolete
    LOCALE_IDEFAULTCOUNTRY = &HA              '  default country code, obsolete
    LOCALE_IDEFAULTCODEPAGE = &HB             '  default OEM code page
    LOCALE_IDEFAULTANSICODEPAGE = &H1004      '  default ANSI codepage, see http://msdn.microsoft.com/en-us/library/windows/desktop/dd317756%28v=vs.85%29.aspx
    LOCALE_IDEFAULTMACCODEPAGE = &H1011       '  default Macintosh codepage associated with the locale

    LOCALE_SLIST = &HC                        '  list item separator
    LOCALE_IMEASURE = &HD                     '  0 = metric, 1 = US

    LOCALE_SDECIMAL = &HE                     '  decimal separator
    LOCALE_STHOUSAND = &HF                    '  thousand separator
    LOCALE_SGROUPING = &H10                   '  digit grouping
    LOCALE_IDIGITS = &H11                     '  number of fractional digits
    LOCALE_ILZERO = &H12                      '  leading zeros for decimal
    LOCALE_INEGNUMBER = &H1010                '  Negative number mode, that is, the format for a negative number, values see http://msdn.microsoft.com/en-us/library/windows/desktop/dd373791%28v=vs.85%29.aspx
    LOCALE_SNATIVEDIGITS = &H13               '  Native equivalents of ASCII 0 through 9, i.e. returns "0123456789"
    LOCALE_IDIGITSUBSTITUTION = &H1014        '  The shape of digits, values see http://msdn.microsoft.com/en-us/library/windows/desktop/dd373769%28v=vs.85%29.aspx
    WV_LOCALE_SNAN = &H69                     '  >=WinVista: Localized name for "Not a number", i.e. "Nan"
    WV_LOCALE_SNEGINFINITY = &H6B             '  >=WinVista: String value for "negative infinity", for example, "-Infinity" for the English (United States) locale
    WV_LOCALE_SPOSINFINITY = &H6A             '  >=WinVista: String value for "positive infinity", for example, "Infinity" for the English (United States) locale.
    LOCALE_SCURRENCY = &H14                   '  local currency symbol
    LOCALE_SINTLSYMBOL = &H15                 '  international currency symbol, i.e. "USD"
    LOCALE_SMONDECIMALSEP = &H16              '  monetary decimal separator
    LOCALE_SMONTHOUSANDSEP = &H17             '  monetary thousand separator
    LOCALE_SMONGROUPING = &H18                '  monetary grouping
    LOCALE_ICURRDIGITS = &H19                 '  number of local currency digits
    LOCALE_IINTLCURRDIGITS = &H1A             '  number of international currency digits
    LOCALE_ICURRENCY = &H1B                   '  positive currency mode
    LOCALE_INEGCURR = &H1C                    '  negative currency mode, values see http://msdn.microsoft.com/en-us/library/windows/desktop/dd373791%28v=vs.85%29.aspx
    LOCALE_SENGCURRNAME = &H1007              '  The full English currency name of the locale
    LOCALE_SNATIVECURRNAME = &H1008           '  The native name of the currency associated with the locale, in the native language of the locale

    LOCALE_SDATE = &H1D                       '  date separator string
    LOCALE_STIME = &H1E                       '  time separator string
    LOCALE_SSHORTDATE = &H1F                  '  short date format string
    LOCALE_SLONGDATE = &H20                   '  long date format string
    LOCALE_STIMEFORMAT = &H1003               '  time format string
    LOCALE_IDATE = &H21                       '  short date format ordering, 0 = M-D-Y, 1 = D-M-Y,2 = Y-M-D
    LOCALE_ILDATE = &H22                      '  long date format ordering, 0 = M-D-Y, 1 = D-M-Y,2 = Y-M-D
    LOCALE_ITIME = &H23                       '  time format specifier, 0=AM/PM, 1 = 24h
    LOCALE_ITIMEMARKPOSN = &H1005             '  time marker string ("AM/PM") specifier 0=suffix, 1=prefix
    WV_LOCALE_SDURATION = &H5D                '  >=WinVista: Time duration format composed of format pictures, values see http://msdn.microsoft.com/en-us/library/windows/desktop/dd373842%28v=vs.85%29.aspx
    LOCALE_ICENTURY = &H24                    '  century format specifier,0=2 digits, 1=4 digits
    LOCALE_ITLZERO = &H25                     '  leading zeros in time fields, 0=no leading zeros for hours,1=leading zeros
    LOCALE_IDAYLZERO = &H26                   '  leading zeros in day fields (short date only), 0=no leading zeros,1=leading zeros
    LOCALE_IMONLZERO = &H27                   '  leading zeros in month fields (short date only), 0=no leading zeros,1=leading zeros
    LOCALE_S1159 = &H28                       '  AM designator string (up to 15 characters since W2003 Server)
    LOCALE_S2359 = &H29                       '  PM designator string (up to 15 characters since W2003 Server)

    LOCALE_ICALENDARTYPE = &H1009             '  Current calendar type
    LOCALE_IOPTIONALCALENDAR = &H100B         '  An optional calendar type that is available for a locale
    LOCALE_IFIRSTDAYOFWEEK = &H100C           '  0 (Monday) - 6 (Sunday)
    LOCALE_IFIRSTWEEKOFYEAR = &H100D          '  first week of the year,
                                              '                0 = Week containing 1/1 is the first week of the year. Note that this can be a single day, if 1/1 falls on the last day of the week
                                              '                1 = First full week following 1/1 is the first week of the year.
                                              '                2 = First week containing at least four days is the first week of the year.
    LOCALE_SDAYNAME1 = &H2A                   '  native long name for Monday    (max. 80 characters)
    LOCALE_SDAYNAME2 = &H2B                   '  native long name for Tuesday   (max. 80 characters)
    LOCALE_SDAYNAME3 = &H2C                   '  native long name for Wednesday (max. 80 characters)
    LOCALE_SDAYNAME4 = &H2D                   '  native long name for Thursday  (max. 80 characters)
    LOCALE_SDAYNAME5 = &H2E                   '  native long name for Friday    (max. 80 characters)
    LOCALE_SDAYNAME6 = &H2F                   '  native long name for Saturday  (max. 80 characters)
    LOCALE_SDAYNAME7 = &H30                   '  native long name for Sunday    (max. 80 characters)
    WV_LOCALE_SSHORTESTDAYNAME1 = &H60        '  >=WinVista: native shortest name for Monday    (max. 80 characters)
    WV_LOCALE_SSHORTESTDAYNAME2 = &H61        '  >=WinVista: native shortest name for Tuesday   (max. 80 characters)
    WV_LOCALE_SSHORTESTDAYNAME3 = &H62        '  >=WinVista: native shortest name for Wednesday (max. 80 characters)
    WV_LOCALE_SSHORTESTDAYNAME4 = &H63        '  >=WinVista: native shortest name for Thursday  (max. 80 characters)
    WV_LOCALE_SSHORTESTDAYNAME5 = &H64        '  >=WinVista: native shortest name for Friday    (max. 80 characters)
    WV_LOCALE_SSHORTESTDAYNAME6 = &H65        '  >=WinVista: native shortest name for Saturday  (max. 80 characters)
    WV_LOCALE_SSHORTESTDAYNAME7 = &H66        '  >=WinVista: native shortest name for Sunday    (max. 80 characters)
    LOCALE_SABBREVDAYNAME1 = &H31             '  native abbreviated name for Monday    (max. 80 characters)
    LOCALE_SABBREVDAYNAME2 = &H32             '  native abbreviated name for Tuesday   (max. 80 characters)
    LOCALE_SABBREVDAYNAME3 = &H33             '  native abbreviated name for Wednesday (max. 80 characters)
    LOCALE_SABBREVDAYNAME4 = &H34             '  native abbreviated name for Thursday  (max. 80 characters)
    LOCALE_SABBREVDAYNAME5 = &H35             '  native abbreviated name for Friday    (max. 80 characters)
    LOCALE_SABBREVDAYNAME6 = &H36             '  native abbreviated name for Saturday  (max. 80 characters)
    LOCALE_SABBREVDAYNAME7 = &H37             '  native abbreviated name for Sunday    (max. 80 characters)
    LOCALE_SMONTHNAME1 = &H38                 '  native long name for January             (max. 80 characters)
    LOCALE_SMONTHNAME2 = &H39                 '  native long name for February            (max. 80 characters)
    LOCALE_SMONTHNAME3 = &H3A                 '  native long name for March               (max. 80 characters)
    LOCALE_SMONTHNAME4 = &H3B                 '  native long name for April               (max. 80 characters)
    LOCALE_SMONTHNAME5 = &H3C                 '  native long name for May                 (max. 80 characters)
    LOCALE_SMONTHNAME6 = &H3D                 '  native long name for June                (max. 80 characters)
    LOCALE_SMONTHNAME7 = &H3E                 '  native long name for July                (max. 80 characters)
    LOCALE_SMONTHNAME8 = &H3F                 '  native long name for August              (max. 80 characters)
    LOCALE_SMONTHNAME9 = &H40                 '  native long name for September           (max. 80 characters)
    LOCALE_SMONTHNAME10 = &H41                '  native long name for October             (max. 80 characters)
    LOCALE_SMONTHNAME11 = &H42                '  native long name for November            (max. 80 characters)
    LOCALE_SMONTHNAME12 = &H43                '  native long name for December            (max. 80 characters)
    LOCALE_SMONTHNAME13 = &H100E              '  native long name for 13th month (if exists) (max. 80 characters)
    LOCALE_SABBREVMONTHNAME1 = &H44           '  native abbreviated name for January      (max. 80 characters)
    LOCALE_SABBREVMONTHNAME2 = &H45           '  native abbreviated name for February     (max. 80 characters)
    LOCALE_SABBREVMONTHNAME3 = &H46           '  native abbreviated name for March        (max. 80 characters)
    LOCALE_SABBREVMONTHNAME4 = &H47           '  native abbreviated name for April        (max. 80 characters)
    LOCALE_SABBREVMONTHNAME5 = &H48           '  native abbreviated name for May          (max. 80 characters)
    LOCALE_SABBREVMONTHNAME6 = &H49           '  native abbreviated name for June         (max. 80 characters)
    LOCALE_SABBREVMONTHNAME7 = &H4A           '  native abbreviated name for July         (max. 80 characters)
    LOCALE_SABBREVMONTHNAME8 = &H4B           '  native abbreviated name for August       (max. 80 characters)
    LOCALE_SABBREVMONTHNAME9 = &H4C           '  native abbreviated name for September    (max. 80 characters)
    LOCALE_SABBREVMONTHNAME10 = &H4D          '  native abbreviated name for October      (max. 80 characters)
    LOCALE_SABBREVMONTHNAME11 = &H4E          '  native abbreviated name for November     (max. 80 characters)
    LOCALE_SABBREVMONTHNAME12 = &H4F          '  native abbreviated name for December     (max. 80 characters)
    LOCALE_SABBREVMONTHNAME13 = &H100F        '  native abbreviated name for 13th month (if exists) (max. 80 characters)

    LOCALE_SPOSITIVESIGN = &H50               '  String value for the positive sign, for example, "+" for the English (United States) locale
    LOCALE_SNEGATIVESIGN = &H51               '  String value for the negative sign, for example, "-" for the English (United States) locale
    LOCALE_IPOSSIGNPOSN = &H52                '  Formatting index for the positive sign in currency values, values see http://msdn.microsoft.com/en-us/library/windows/desktop/dd373791%28v=vs.85%29.aspx
    LOCALE_INEGSIGNPOSN = &H53                '  Formatting index for the negative sign in currency values, values see http://msdn.microsoft.com/en-us/library/windows/desktop/dd373791%28v=vs.85%29.aspx
    LOCALE_IPOSSYMPRECEDES = &H54             '  Position of the monetary symbol in a positive monetary value, 0=follows value, 1=precedes value
    LOCALE_IPOSSEPBYSPACE = &H55              '  Separation of monetary symbol in a positive monetary value, 0=no space, 1=space
    LOCALE_INEGSYMPRECEDES = &H56             '  Position of the monetary symbol in a negative monetary value, 0=follows value, 1=precedes value
    LOCALE_INEGSEPBYSPACE = &H57              '  Separation of monetary symbol in a negative monetary value, 0=no space, 1=space


    LOCALE_IDEFAULTEBCDICCODEPAGE = &H1012    '  default ebcdic code page
    LOCALE_IPAPERSIZE = &H100A                '  default paper size of the locale, examples: 1=US Letter, 5=US Legal, 8=A3, 9=A4
    W7_LOCALE_SNATIVEDISPLAYNAME = &H73       '  >=Win7: Display name in native locale language, i.e. "Deutsch (Deutschland)"
    LOCALE_SYEARMONTH = &H1006                '  The year-month formatting string for the locale
    LOCALE_SSORTNAME = &H1013                 '  The full localized name of the sort for the specified locale identifier, dependent on the language of the shell. This constant is used to determine casing and sorting behavior, i.e. "Default" or "Dictionary"
    W7_LOCALE_SSORTLOCALE = &H7B              '  >=Win7: Name of the locale to use for sorting or casing behavior
    W7_LOCALE_SLOCALIZEDDISPLAYNAME = &H72    '  >=Win7: Full localized name of the locale for the user interface language, for example, Deutsch (Deutschland) for German (Germany)"

    WV_LOCALE_SSCRIPTS = &H6C                 '  >=WinVista: A string representing a list of scripts, using the 4-character notation used in ISO 15924, i.e. "en-US" or "de-DE"
    WV_LOCALE_SPARENT = &H6D                  '  >=WinVista: Fallback locale, used by the resource loader, i.e. "en-US" falls back to "en" if "en-US" specific is not available
    WV_LOCALE_SCONSOLEFALLBACKNAME = &H6E     '  >=WinVista:  Preferred locale to use for console display

    LOCALE_IPOSITIVEPERCENT = &H75            '  >=Win7: Positive percentage formatting pattern for the locale, 0="# %",1="#%", 2="%#", 3="% #"
    LOCALE_RETURN_NUMBER = &H20000000         '  return a number instead of a string, must be combined with other locale values
    LOCALE_RETURN_GENITIVE_NAMES = &H10000000 '  >=Win7: Retrieve the genitive names of months, which are the names used when the month names are combined with other items, must be combined with other locale values
    LOCALE_IREADINGLAYOUT = &H70              '  >=Win7: The reading layout for text, 0=Left to right, 1=Right to left, 2=Top to bottom and right to left, 3=Top to bottom and left to right

End Enum

'#######################################################################################
'---------------------------------------------------------------------------------------
'--------------------------------- Class Variables  ------------------------------------
'---------------------------------------------------------------------------------------
'#######################################################################################

Private prv_varDayReplace() As Variant
Private prv_varDays() As Variant
Private prv_varSeparators() As Variant
Private prv_varMonthNames() As Variant
Private prv_varAMPMs() As Variant
Private prv_varTestLocales() As Variant
Private prv_varAllLocales() As Variant
Private prv_varAllLocaleNames() As Variant
Private prv_varAllCountries() As Variant
Private prv_varPaperSizes() As Variant
Private prv_datDate As Date
Private prv_typWinVersion As typWinVersionInfo
Private prv_lngWinVersion As EnmCCRegionalsWinVersion

'#######################################################################################
'---------------------------------------------------------------------------------------
'-------------------------------------- APIs  ------------------------------------------
'---------------------------------------------------------------------------------------
'#######################################################################################

Private Declare Function apiGetLocaleInfo Lib "kernel32" Alias "GetLocaleInfoA" _
                                         (ByVal lngLocale As Long, _
                                          ByVal lngLocaleType As Long, ByVal strLocaleDataBuffer As String, _
                                          ByVal lngDataBufferSize As Long) As Long


Private Declare Function GetVersionEx Lib "kernel32" Alias "GetVersionExA" _
                                          (ByRef VersionInformation As typWinVersionInfo) As Long

'#######################################################################################
'---------------------------------------------------------------------------------------
'-----------------------------------   Properties  -------------------------------------
'---------------------------------------------------------------------------------------
'#######################################################################################

'---------------------------------------------------------------------------------------
' Property    : WinVersion
' Access Meth.: Read Only
' Date        : 10.02.2013
' Purpose     : Returns WinXP, WinVista or Win7 as enum value (everything lower than Windows XP will be handled as Windows XP)
' Returns     : enm - EnmCCRegionalsWinVersion
'---------------------------------------------------------------------------------------
'
Public Property Get WinVersion() As EnmCCRegionalsWinVersion
    WinVersion = prv_lngWinVersion
End Property

'---------------------------------------------------------------------------------------
' Property     : DateAndTimeValue
' Date         : 29.01.2013
' Purpose      : Returns the value in prv_datDate
' Requirements : fnValidateDateAndTime must be used before
' Returns      : dat - Date
'---------------------------------------------------------------------------------------
'
Public Property Get DateAndTimeValue() As Date
    DateAndTimeValue = prv_datDate
End Property

'---------------------------------------------------------------------------------------
' Property     : DateAndTimeString
' Date         : 29.01.2013
' Purpose      : Returns the value in prv_datDate as formatted string in the chosen locale
' Requirements : fnValidateDateAndTime must be used before
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get DateAndTimeString(Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_INVARIANT, _
                                      Optional ByVal intFormat As EnmCCRegionalsDateFormat = enmCCRegDateFormat_Short) As String
    DateAndTimeString = Me.DateOnlyString(lngLocaleType, intFormat) & " " & _
                        Me.TimeOnlyString(lngLocaleType)
End Property

'---------------------------------------------------------------------------------------
' Property     : DateOnlyString
' Date         : 29.01.2013
' Purpose      : Returns the date part of the value in prv_datDate as formatted string in the chosen locale
' Requirements : fnValidateDateAndTime must be used before
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get DateOnlyString(Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_INVARIANT, _
                                   Optional ByVal intFormat As EnmCCRegionalsDateFormat = enmCCRegDateFormat_Short) As String
    Dim strDateFormat As String
    Dim strDateSeparator As String
    Dim lngType As Long
    
    Select Case intFormat
        Case enmCCRegDateFormat_Short
            lngType = LOCALE_SSHORTDATE
        Case enmCCRegDateFormat_Long
            lngType = LOCALE_SLONGDATE
        Case enmCCRegDateFormat_YearMonth
            lngType = LOCALE_SYEARMONTH
    End Select
    
    strDateFormat = Me.fnLocaleInfo(lngType, lngLocaleType)
    strDateSeparator = Me.fnLocaleInfo(LOCALE_SDATE, lngLocaleType)
    
    strDateFormat = Replace(strDateFormat, strDateSeparator, "\" & strDateSeparator)
    strDateFormat = Format(Me.DateOnlyValue, strDateFormat)
    strDateFormat = Replace(strDateFormat, Me.fnMonthNames(LOCALENAME_Long, A_LOCALE_USER_DEFAULT)(Month(Me.DateOnlyValue)), _
                                           Me.fnMonthNames(LOCALENAME_Long, lngLocaleType)(Month(Me.DateOnlyValue)))
    strDateFormat = Replace(strDateFormat, Me.fnMonthNames(LOCALENAME_short, A_LOCALE_USER_DEFAULT)(Month(Me.DateOnlyValue)), _
                                           Me.fnMonthNames(LOCALENAME_short, lngLocaleType)(Month(Me.DateOnlyValue)))
    strDateFormat = Replace(strDateFormat, Me.fnDayNames(LOCALENAME_Long, A_LOCALE_USER_DEFAULT)(Weekday(Me.DateOnlyValue, vbMonday)), _
                                           Me.fnDayNames(LOCALENAME_Long, lngLocaleType)(Weekday(Me.DateOnlyValue, vbMonday)))
    DateOnlyString = Replace(strDateFormat, Me.fnDayNames(LOCALENAME_short, A_LOCALE_USER_DEFAULT)(Weekday(Me.DateOnlyValue, vbMonday)), _
                                           Me.fnDayNames(LOCALENAME_short, lngLocaleType)(Weekday(Me.DateOnlyValue, vbMonday)))
End Property

'---------------------------------------------------------------------------------------
' Property     : DateOnlyValue
' Date         : 29.01.2013
' Purpose      : Returns the date part of the value in prv_datDate
' Requirements : fnValidateDateAndTime must be used before
' Returns      : dat - Date
'---------------------------------------------------------------------------------------
'
Public Property Get DateOnlyValue() As Date
    DateOnlyValue = Int(CDbl(prv_datDate))
End Property

'---------------------------------------------------------------------------------------
' Property     : TimeOnlyValue
' Date         : 29.01.2013
' Purpose      : Returns the time part of the value in prv_datDate
' Requirements : fnValidateDateAndTime must be used before
' Returns      : dat - Date
'---------------------------------------------------------------------------------------
'
Public Property Get TimeOnlyValue() As Date
    TimeOnlyValue = CDbl(prv_datDate) - Me.DateOnlyValue
End Property

'---------------------------------------------------------------------------------------
' Property     : TimeOnlyString
' Date         : 29.01.2013
' Purpose      : Returns the time part of the value in prv_datDate as formatted time string in the chosen locale
' Requirements : fnValidateDateAndTime must be used before
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get TimeOnlyString(Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_INVARIANT) As String
    Dim strTimeFormat As String
    Dim strTimeSeparator As String
    Dim strAMFormat As String
    Dim strPMFormat As String
    Dim strFormat As String
    Dim i As Integer
    
    strTimeFormat = Me.fnLocaleInfo(LOCALE_STIMEFORMAT, lngLocaleType)
    If lngLocaleType = A_LOCALE_INVARIANT Then
        strTimeFormat = Me.fnLocaleInfo(LOCALE_STIMEFORMAT, A_LOCALE_INVARIANT)
        strTimeSeparator = Me.fnLocaleInfo(LOCALE_STIME, A_LOCALE_INVARIANT)
    Else
        strTimeSeparator = Me.fnLocaleInfo(LOCALE_STIME, lngLocaleType)
        strAMFormat = Me.fnLocaleInfo(LOCALE_S1159, lngLocaleType)
        strPMFormat = Me.fnLocaleInfo(LOCALE_S2359, lngLocaleType)
    End If
        
    strTimeFormat = Replace(strTimeFormat, strTimeSeparator, "\" & strTimeSeparator)
    
    If lngLocaleType <> A_LOCALE_INVARIANT Then
        strFormat = ""
        For i = 1 To Len(strAMFormat)
            strFormat = strFormat & "\" & Mid(strAMFormat, i, 1)
        Next
        strAMFormat = strFormat
        strFormat = ""
        For i = 1 To Len(strPMFormat)
            strFormat = strFormat & "\" & Mid(strPMFormat, i, 1)
        Next
        strPMFormat = strFormat
        
        Do While InStr(1, strTimeFormat, "tt", vbBinaryCompare) > 0
            strTimeFormat = Replace(strTimeFormat, "tt", _
                                    IIf(Hour(prv_datDate) >= 0 And Hour(prv_datDate) <= 11, strAMFormat, strPMFormat))
        Loop
        Do While InStr(1, strTimeFormat, "t", vbBinaryCompare) > 0
            strTimeFormat = Replace(strTimeFormat, "t", _
                                    IIf(Hour(prv_datDate) >= 0 And Hour(prv_datDate) <= 11, Left(strAMFormat, 1), Left(strPMFormat, 1)))
        Loop
    End If
    
    TimeOnlyString = Format(Me.TimeOnlyValue, strTimeFormat)
End Property

'---------------------------------------------------------------------------------------
' Property     : SQLDateOnly
' Date         : 29.01.2013
' Purpose      : Converts a date value into SQL usable standard date string
' Parameters   : intTarget: Access or SQL Server output format
'                bolWithSingleQuotes: For SQL Server only: Surrounds the output string with single quotes
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get SQLDateOnly(Optional ByVal intTarget As EnmCCRegionalsTarget = enmCCRegTarget_Access, _
                                Optional ByVal bolWithSinqleQuotes As Boolean = True) As String
    Select Case intTarget
        Case enmCCRegTarget_Access
            SQLDateOnly = "#" & Me.DateOnlyString & "#"
        Case enmCCRegTarget_SQLServer
            SQLDateOnly = IIf(bolWithSinqleQuotes, "'", "") & _
                          Format(prv_datDate, cFORMAT_SQLDateOnly) & _
                          IIf(bolWithSinqleQuotes, "'", "")
    End Select
End Property

'---------------------------------------------------------------------------------------
' Property     : SQLTimeOnly
' Date         : 29.01.2013
' Purpose      : Converts a date value into SQL usable standard time string
' Parameters   : intTarget: Access or SQL Server output format
'                bolWithSingleQuotes: For SQL Server only: Surrounds the output string with single quotes
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get SQLTimeOnly(Optional ByVal intTarget As EnmCCRegionalsTarget = enmCCRegTarget_Access, _
                                Optional ByVal bolWithSinqleQuotes As Boolean = True) As String
    Select Case intTarget
        Case enmCCRegTarget_Access
            SQLTimeOnly = "#" & Me.TimeOnlyString & "#"
        Case enmCCRegTarget_SQLServer
            SQLTimeOnly = IIf(bolWithSinqleQuotes, "'", "") & _
                          Format(prv_datDate, cFORMAT_SQLTimeOnly) & _
                          IIf(bolWithSinqleQuotes, "'", "")
    End Select
End Property

'---------------------------------------------------------------------------------------
' Property     : SQLDateAndTime
' Date         : 29.01.2013
' Purpose      : Converts a date value into SQL usable standard date string
' Parameters   : intTarget: Access or SQL Server output format
'                bolWithSingleQuotes: For SQL Server only: Surrounds the output string with single quotes
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get SQLDateAndTime(Optional ByVal intTarget As EnmCCRegionalsTarget = enmCCRegTarget_Access, _
                                   Optional ByVal bolWithSinqleQuotes As Boolean = True) As String
    Select Case intTarget

        Case enmCCRegTarget_Access
            SQLDateAndTime = "#" & Me.DateOnlyString & " " & Me.TimeOnlyString & "#"
        Case enmCCRegTarget_SQLServer
            SQLDateAndTime = IIf(bolWithSinqleQuotes, "'", "") & _
                             Format(prv_datDate, cFORMAT_SQLDateAndTime) & _
                             IIf(bolWithSinqleQuotes, "'", "")
    End Select
End Property

'---------------------------------------------------------------------------------------
' Property    : LocaleID
' Access Meth.: Read Only
' Date        : 29.05.2013
' Purpose     : Get the locale ID from the "All Locales" array with the chosen index
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Property Get LocaleID(lngIndex As Long) As String
    If lngIndex > UBound(prv_varAllLocales) Then
        LocaleID = ""
    Else
        LocaleID = prv_varAllLocales(lngIndex)
    End If
End Property

'---------------------------------------------------------------------------------------
' Property    : AllLocales
' Access Meth.: Read Only
' Date        : 29.05.2013
' Purpose     : Get the variant array with the list of all locale IDs
' Returns     : var - Variant
'---------------------------------------------------------------------------------------
'
Public Property Get AllLocales() As Variant()
    AllLocales = prv_varAllLocales
End Property

'---------------------------------------------------------------------------------------
' Property    : AllLocaleNames
' Access Meth.: Read Only
' Date        : 29.05.2013
' Purpose     : Get the variant array with the list of all locale names (English)
' Returns     : var - Variant
'---------------------------------------------------------------------------------------
'
Public Property Get AllLocaleNames() As Variant()
    AllLocaleNames = prv_varAllLocaleNames
End Property

'---------------------------------------------------------------------------------------
' Property    : PaperSize
' Access Meth.: Read Only
' Date        : 29.05.2013
' Purpose     : Get the default paper size name using the intIndex parameter (equal to the Windows locale "LOCALE_IPAPERSIZE")
' Returns     : var - Variant
'---------------------------------------------------------------------------------------
'
Public Property Get PaperSize(intIndex As Integer) As String
    PaperSize = prv_varPaperSizes(intIndex)
End Property

'---------------------------------------------------------------------------------------
' Property    : UserPaperSize
' Access Meth.: Read Only
' Date        : 29.05.2013
' Purpose     : Get the default paper size name using the current user locale
' Returns     : var - Variant
'---------------------------------------------------------------------------------------
'
Public Property Get UserPaperSize() As String
    UserPaperSize = prv_varPaperSizes(Me.fnLocaleInfo(LOCALE_IPAPERSIZE, A_LOCALE_USER_DEFAULT))
End Property

'#######################################################################################
'---------------------------------------------------------------------------------------
'--------------------------------------- Methods ---------------------------------------
'---------------------------------------------------------------------------------------
'#######################################################################################

'---------------------------------------------------------------------------------------
' Property     : fnLocaleInfo
' Date         : 23.01.2013
' Purpose      : Returns the specified value from Windows Regional Settings
' Parameter    : lngLocaleSettingType : Specifies the desired value to be returned (see Enum description above)
'                lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other desired language)
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Function fnLocaleInfo(ByVal lngLocaleSettingType As EnmCCRegionals, _
                             Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_USER_DEFAULT) As String
    Dim strLocaleBuffer As String
    Dim lngLength As Long

   On Error GoTo fnLocaleInfo_Error

    fnLocaleInfo = ""

    If lngLocaleType = 0 Then lngLocaleType = A_LOCALE_INVARIANT
    ' Get the needed length of the buffer only
    lngLength = apiGetLocaleInfo(lngLocaleType, lngLocaleSettingType, strLocaleBuffer, 0)
    If lngLength <> 0 Then
        ' Set the buffer length
        strLocaleBuffer = String(lngLength + 1, " ")
        ' Now return the wanted value
        lngLength = apiGetLocaleInfo(lngLocaleType, lngLocaleSettingType, strLocaleBuffer, lngLength)
        If lngLength <> 0 Then fnLocaleInfo = Left(strLocaleBuffer, lngLength - 1)
    End If

fnLocaleInfo_Exit:
    Exit Function

fnLocaleInfo_Error:
    Select Case Err
        Case Else
            Resume fnLocaleInfo_Exit
    End Select
End Function

'---------------------------------------------------------------------------------------
' Property     : fnVal
' Date         : 23.01.2013
' Purpose      : Converts a string into a double value (replacement for Val() and CDbl())
'                Deletes all characters "strThousand" from the input string.
'                Replaces characters "strDecimal" with a dot and then uses Val() to convert
' Parameters   : strValue   : String to be converted
'                strThousand: String for the thousand character ("," in US, "." in German for example)
'                strDecimal : String for the decimal character to be exchanged by dot.
'                If both parameters strThousand and strDecimal are empty the function fnValLocale
'                is called first which gets the Windows Regional Settings to specify the
'                characters for thousand and decimal.
' Returns      : dbl - Double
'---------------------------------------------------------------------------------------
'
Public Function fnVal(ByVal strValue As String, Optional ByVal strThousand As String = "", _
                                                Optional ByVal strDecimal As String = "") As Double
    
   On Error GoTo fnVal_Error

    If strThousand = "" And strDecimal = "" Then
        fnVal = Me.fnValLocale(strValue)
        Exit Function
    End If
    
    strValue = Replace(strValue, strThousand, "")
    strValue = Replace(strValue, strDecimal, ".")
    ' Replace multiple decimal separator characters with single ones
    Do Until strValue = Replace(strValue, strDecimal & strDecimal, ".")
        strValue = Replace(strValue, strDecimal & strDecimal, ".")
    Loop
    
    fnVal = Val(strValue)

fnVal_Exit:
    Exit Function

fnVal_Error:
    Select Case Err
        Case Else
            Resume fnVal_Exit
    End Select

End Function

'---------------------------------------------------------------------------------------
' Property     : fnValParse
' Date         : 23.01.2013
' Purpose      : Tries to parse the input string so that the result is a valid number for Val()
' Example      : Val() would change the German "1.000.000,23" into "1", CDbl would throw
'                a type mismatch error if the Windows Regional Settings are not German (or compatible).
'                fnValParse returns "1000000.23" independent of the regional settings by
'                testing which character comes first, the one for thousand or decimal.
'                So if the US format "1,000,000.23" is used, Val() would also return "1",
'                CDbl would return type mismatch if Windows Regional Settings are not US (or compatible)
'                but fnValParse would also return "1000000.23".
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Function fnValParse(ByVal strValue As String) As String
    Dim strLetter As String
    Dim strOutput As String
    Dim i As Long
    Dim bolFirstAlphaFound As Boolean
    Dim strThousand As String
    Dim strDecimal As String
    Dim lngThousandPos As Long
    Dim lngDecimalPos As Long
    
   On Error GoTo fnValParse_Error

    bolFirstAlphaFound = False
    strValue = Trim(strValue)
    ' Replace multiple ",," or ".." with single ones
    Do Until strValue = Replace(strValue, "..", ".")
        strValue = Replace(strValue, "..", ".")
    Loop
    Do Until strValue = Replace(strValue, ",,", ",")
        strValue = Replace(strValue, ",,", ",")
    Loop
    
    For i = 1 To Len(strValue)
        strLetter = Mid(strValue, i, 1)
        Select Case strLetter
            Case "0", "1", "2", "3", "4", "5", "6", "7", "8", "9", ",", ".", "-", "+"
                If Not bolFirstAlphaFound Then
                    If strLetter = "+" Then strLetter = ""
                    If i > 1 And strLetter = "-" Then strLetter = ""
                    strOutput = strOutput & strLetter
                End If
            Case Else
                bolFirstAlphaFound = True
        End Select
    Next
    
    lngDecimalPos = InStrRev(strOutput, ",")
    lngThousandPos = InStrRev(strOutput, ".")
    If lngDecimalPos > lngThousandPos Then
        strThousand = "."
        strDecimal = ","
    Else
        strThousand = ","
        strDecimal = "."
    End If
    
    strOutput = Replace(strOutput, strThousand, "")
    strOutput = Replace(strOutput, strDecimal, ".")
    
    fnValParse = strOutput

fnValParse_Exit:
    Exit Function

fnValParse_Error:
    Select Case Err
        Case Else
            Resume fnValParse_Exit
    End Select
End Function

'---------------------------------------------------------------------------------------
' Property     : fnValLocale
' Date         : 23.01.2013
' Purpose      : Checks the Windows Regional Settings for decimal and thousand character
'                and deletes the thousand separator from the string and replaces the decimal
'                character with a dot. So the German "1.000,23" would return "1000.23" and
'                then the Val() function is used to convert the string to a double.
' Parameters   : strValue: Value-String to convert to Double value
'                lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other desired language)
' Returns      : dbl - Double
'---------------------------------------------------------------------------------------
'
Public Function fnValLocale(ByVal strValue As String, _
                            Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_USER_DEFAULT) As Double
    Dim strThousand As String
    Dim strDecimal As String
    
   On Error GoTo fnVal_Error
    
    strThousand = fnLocaleInfo(LOCALE_STHOUSAND, lngLocaleType)
    strDecimal = fnLocaleInfo(LOCALE_SDECIMAL, lngLocaleType)
       
    ' Replace multiple decimal separator characters with single ones
    Do Until strValue = Replace(strValue, strDecimal & strDecimal, strDecimal)
        strValue = Replace(strValue, strDecimal & strDecimal, strDecimal)
    Loop
    Do Until strValue = Replace(strValue, strThousand & strThousand, strThousand)
        strValue = Replace(strValue, strThousand & strThousand, strThousand)
    Loop
    
    strValue = Replace(strValue, strThousand, "")
    strValue = Replace(strValue, strDecimal, ".")
    fnValLocale = Val(strValue)
    
fnVal_Exit:
    Exit Function

fnVal_Error:
    Select Case Err
        Case Else
            Resume fnVal_Exit
    End Select
    
End Function

'---------------------------------------------------------------------------------------
' Property     : fnMonthNames
' Date         : 23.01.2013
' Purpose      : Returns the month names using the Windows Regional Settings
' Parameters   : intShortOrLong: Specifies if the three character name is returned or the complete month name
'                lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other desired language)
' Returns      : String Array (1 to 12) containing the month names
'---------------------------------------------------------------------------------------
'
Public Function fnMonthNames(Optional ByVal intShortOrLong As EnmCCRegionalsShortOrLong = LOCALENAME_Long, _
                             Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_USER_DEFAULT) As String()
    Dim strMonths(1 To 12) As String
    Dim i As Byte
    
    If intShortOrLong < LOCALENAME_short Or intShortOrLong > LOCALENAME_Shortest Then intShortOrLong = LOCALENAME_Long
    
    For i = 0 To 11
        strMonths(i + 1) = Me.fnLocaleInfo(Choose(intShortOrLong + 1, _
                                                                    LOCALE_SMONTHNAME1 + i, _
                                                                    LOCALE_SABBREVMONTHNAME1 + i, _
                                                                    LOCALE_SABBREVMONTHNAME1 + i), lngLocaleType)
        If intShortOrLong = LOCALENAME_Shortest Then strMonths(i + 1) = Left(strMonths(i + 1), 1)
    Next
    fnMonthNames = strMonths
End Function

'---------------------------------------------------------------------------------------
' Property     : fnMonthNamesCSV
' Date         : 23.01.2013
' Purpose      : Like above but returns the names as comma separated string (CSV).
' Parameters   : intShortOrLong: Specifies if the three character name is returned or the complete month name
'                strNameDelimiter: Adds the character to each name (before and after the name)
'                strDelimiter: Delimiter character, default is comma
'                lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other desired language)
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Function fnMonthNamesCSV(Optional ByVal intShortOrLong As EnmCCRegionalsShortOrLong = LOCALENAME_Long, _
                                Optional ByVal strNameDelimiter As String = "", _
                                Optional ByVal strDelimiter As String = ",", _
                                Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_USER_DEFAULT) As String
    Dim strMonths() As String
    Dim i As Byte
    
    strMonths = Me.fnMonthNames(intShortOrLong, lngLocaleType)
    If strNameDelimiter <> "" Then
        For i = 1 To 12
            strMonths(i) = strNameDelimiter & strMonths(i) & strNameDelimiter
        Next
    End If
    
    fnMonthNamesCSV = Join(strMonths, strDelimiter)
End Function

'---------------------------------------------------------------------------------------
' Property     : fnDayNames
' Date         : 23.01.2013
' Purpose      : Returns the weekday names using the Windows Regional Settings
' Parameters   : intShortOrLong: Specifies if the three character name is returned or the complete day name
'                lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other desired language)
' Returns      : String Array (1 to 7) containing the day names
'---------------------------------------------------------------------------------------
'
Public Function fnDayNames(Optional ByVal intShortOrLong As EnmCCRegionalsShortOrLong = LOCALENAME_Long, _
                           Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_USER_DEFAULT) As String()
    Dim strDays(1 To 7) As String
    Dim i As Byte
    
    If intShortOrLong < LOCALENAME_short Or intShortOrLong > LOCALENAME_Shortest Then intShortOrLong = LOCALENAME_Long
    
    For i = 0 To 6
        strDays(i + 1) = Me.fnLocaleInfo(Choose(intShortOrLong + 1, _
                                                                    LOCALE_SDAYNAME1 + i, _
                                                                    LOCALE_SABBREVDAYNAME1 + i, _
                                                                    LOCALE_SABBREVDAYNAME1 + i), _
                                         lngLocaleType)
        If intShortOrLong = LOCALENAME_Shortest Then
            strDays(i + 1) = Left(Me.fnLocaleInfo(LOCALE_SDAYNAME1 + i, lngLocaleType), 1)
        End If
    Next
    fnDayNames = strDays
End Function

'---------------------------------------------------------------------------------------
' Property     : fnDayNamesCSV
' Date         : 23.01.2013
' Purpose      : Like above but returns the names as comma separated string (CSV).
' Parameters   : intShortOrLong: Specifies if the three character name is returned or the complete day name
'                strNameDelimiter: Adds the character to each name (before and after the name)
'                strDelimiter: Delimiter character, default is comma
'                lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other desired language)
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Function fnDayNamesCSV(Optional ByVal intShortOrLong As EnmCCRegionalsShortOrLong = LOCALENAME_Long, _
                              Optional ByVal strNameDelimiter As String = "", _
                              Optional ByVal strDelimiter As String = ",", _
                              Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_USER_DEFAULT) As String
    Dim strDays() As String
    Dim i As Byte
    
    strDays = Me.fnDayNames(intShortOrLong, lngLocaleType)
    If strNameDelimiter <> "" Then
        For i = 1 To 7
            strDays(i) = strNameDelimiter & strDays(i) & strNameDelimiter
        Next
    End If
    
    fnDayNamesCSV = Join(strDays, strDelimiter)
End Function

'---------------------------------------------------------------------------------------
' Property     : fnDaysCount
' Date         : 29.01.2013
' Purpose      : Returns a byte array containing the number of days for each month
'                intYear: if supplied, the month February will be calculated with this year, otherwise the current year is used
' Returns      : Byte Array
'---------------------------------------------------------------------------------------
'
Public Function fnDaysCount(Optional ByVal intYear As Integer = -1) As Byte()
    Dim i As Byte
    Dim bytOutput(1 To 12) As Byte
    
    For i = 1 To 12
        bytOutput(i) = prv_varDays(i - 1)
    Next
    
    If intYear < 0 Then intYear = Year(VBA.Date())
    If Me.fnLeapYear(intYear) Then bytOutput(2) = bytOutput(2) + 1
    
    fnDaysCount = bytOutput
End Function

'---------------------------------------------------------------------------------------
' Property     : fnDaysCountCSV
' Date         : 29.01.2013
' Purpose      : Returns a string containing the number of days for each month separated by comma
'                intYear: if supplied, the month February will be calculated with this year, otherwise the current year is used
'                strNameDelimiter: Adds the character to each value (before and after the value)
'                strDelimiter: Delimiter character, default is comma
' Returns      : str - String
'---------------------------------------------------------------------------------------
'
Public Function fnDaysCountCSV(Optional ByVal intYear As Integer = -1, _
                               Optional ByVal strNameDelimiter As String = "", _
                               Optional ByVal strDelimiter As String = ",") As String
    Dim bytDaysCount() As Byte
    Dim strDaysCount(1 To 12) As String
    Dim i As Byte
    
    bytDaysCount = Me.fnDaysCount(intYear)
    For i = 1 To 12
        If strNameDelimiter <> "" Then
            strDaysCount(i) = strNameDelimiter & bytDaysCount(i) & strNameDelimiter
        Else
            strDaysCount(i) = bytDaysCount(i)
        End If
    Next
    
    fnDaysCountCSV = Join(strDaysCount, strDelimiter)
End Function

'---------------------------------------------------------------------------------------
' Procedure   : fnCountries
' Date        : 10.02.2013
' Purpose     : Returns an array of all countries known in Windows in different formats
' Parameters  : lngOutputType: Wanted kind of output like defined in the enum
' Returns     : var - Variant array
'---------------------------------------------------------------------------------------
'
Public Function fnCountries(Optional ByVal lngOutputType As EnmCCRegionalsCountryOutputType = enmCCRegOutputType_LocalNameLong, _
                            Optional ByVal bolSort As Boolean = False) As String()
    Dim strCountries() As String
    Dim varExclude() As Variant
    Dim lngOutput As Long
    Dim i As Long, k As Long, m As Long
    
'    varExclude = Array(20, 24, 45, 46, 56, 57, 60, 61, 62, 65, 66, 69, 74, 80, 83, 85, 89, 91, 93, 94, 96, 97, 101, 102, 121, 124, 125, 126, 129, 133, 135, 136, 137) ' double locales or languages only
'    varExclude = Array(0, 0)
    
    
    k = 0
    For i = 0 To UBound(prv_varAllCountries)
'        For m = 0 To UBound(varExclude)
'            If i = varExclude(m) Then Exit For
'        Next
'        If m = UBound(varExclude) + 1 Then
            ReDim Preserve strCountries(k)
            Select Case lngOutputType

                Case enmCCRegOutputType_ISO3166
                    Select Case prv_lngWinVersion
                        Case Version_WinXP
                            lngOutput = LOCALE_SISO3166CTRYNAME
                        Case Version_WinVista, Version_Win7
                            lngOutput = WV_LOCALE_SISO3166CTRYNAME2
                    End Select
                Case enmCCRegOutputType_ibm
                    lngOutput = LOCALE_ICOUNTRY
                Case enmCCRegOutputType_GeoID
                    lngOutput = LOCALE_IGEOID
                Case enmCCRegOutputType_LocalNameLong
                    Select Case prv_lngWinVersion
                        Case Version_WinXP, Version_WinVista
                            lngOutput = LOCALE_SCOUNTRY
                        Case Version_Win7
                            lngOutput = W7_LOCALE_SLOCALIZEDCOUNTRYNAME
                    End Select
                Case enmCCRegOutputType_NativeNameLong
                    Select Case prv_lngWinVersion
                        Case Version_WinXP, Version_WinVista
                            lngOutput = LOCALE_SNATIVECTRYNAME
                        Case Version_Win7
                            lngOutput = W7_LOCALE_SNATIVECOUNTRYNAME
                    End Select
                Case enmCCRegOutputType_EnglishNamesLong
                    lngOutput = LOCALE_SENGCOUNTRY
                Case enmCCRegOutputType_AbbreviatedName
                    lngOutput = LOCALE_SABBREVCTRYNAME
                Case enmCCRegOutputType_Currency
                    lngOutput = LOCALE_SENGCURRNAME
                Case enmCCRegOutputType_CurrencyISO
                    lngOutput = LOCALE_SINTLSYMBOL
            End Select
            
            Select Case lngOutputType
'                Case enmCCRegOutputType_DefaultLocaleNames
'                    strCountries(k) = prv_varAllLocaleNames(i)
                Case enmCCRegOutputType_Index
                    strCountries(k) = i
                Case enmCCRegOutputType_LocaleID
                    strCountries(k) = prv_varAllCountries(i)
                Case Else
                    strCountries(k) = Me.fnLocaleInfo(lngOutput, prv_varAllCountries(i))
                    If lngOutputType = enmCCRegOutputType_LocalNameLong Then
                        strCountries(k) = Replace(strCountries(k), "?", "")
                    End If
            End Select
            k = k + 1
'        End If
    Next
    If bolSort Then Me.Mischsort strCountries
    fnCountries = strCountries
End Function

'---------------------------------------------------------------------------------------
' Procedure   : fnCountriesCSV
' Date        : 10.02.2013
' Purpose     : Returns the country names as comma separated list
' Parameters  : lngOutputType: Wanted kind of output like defined in the enum
'               strNameDelimiter: Adds the character to each name (before and after the name)
'               strDelimiter: Delimiter character, default is comma
' Returns     : str - String
'---------------------------------------------------------------------------------------
'
Public Function fnCountriesCSV(Optional ByVal lngOutputType As EnmCCRegionalsCountryOutputType = enmCCRegOutputType_LocalNameLong, _
                               Optional ByVal strNameDelimiter As String = "", _
                               Optional ByVal strDelimiter As String = ",", _
                               Optional ByVal bolSort As Boolean = False) As String
    Dim strCountries() As String
    Dim i As Long
    
    strCountries = fnCountries(lngOutputType, bolSort)
    If strNameDelimiter <> "" Then
        For i = 0 To UBound(strCountries)
            strCountries(i) = strNameDelimiter & strCountries(i) & strNameDelimiter
        Next
    End If
    fnCountriesCSV = Join(strCountries, strDelimiter)
End Function

'---------------------------------------------------------------------------------------
' Procedure   : fnCountryValueList
' Date        : 10.02.2013
' Purpose     : Returns a string which can be used as value list property for comboboxes/listboxes
' Parameters  : lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other wanted language)
'               bolSort: Sort the output (if only one column specified, sort this, otherwise sort by column 1 (index begins with 0)
'                        so the returned list sorts by the display column of a combobox
'               lngOutputType: a value list from EnmCCRegionalsOutputType to specify the desired column(s)
' Returns     : var - Variant
'---------------------------------------------------------------------------------------
'
Public Function fnCountryValueList(ByVal lngLocaleType As EnmCCRegionalsType, _
                                   ByVal bolSort As Boolean, _
                                   ParamArray lngOutputType() As Variant) As String
    Dim strDelimiter As String
    Dim strOutput As String
    Dim varTable() As Variant
    Dim strTemp() As String
    Dim varArr() As Variant
    Dim strArr() As String
    Dim i As Long, k As Long
    strDelimiter = Me.fnListSeparator(lngLocaleType)
    
    If UBound(lngOutputType) > -1 Then
        ReDim varTable(UBound(lngOutputType))
        For i = 0 To UBound(lngOutputType)
            varTable(i) = Me.fnCountries(lngOutputType(i))
        Next
        For k = 0 To UBound(varTable(0))
            For i = 0 To UBound(lngOutputType)
                strOutput = strOutput & """" & varTable(i)(k) & """" & IIf(i < UBound(lngOutputType), strDelimiter, "")
            Next
        Next
        If Len(strOutput) > 1 And right(strOutput, 1) = "" Then strOutput = Left(strOutput, Len(strOutput) - 1)
        strTemp = Split(strOutput, "")
        If UBound(lngOutputType) > 0 Then
            ReDim varArr(UBound(varTable(0)))
            ReDim strArr(UBound(varTable(0)))
            For i = 0 To UBound(varTable(0))
                varArr(i) = varTable(1)(i) & "," & i
            Next
            If bolSort Then Me.Mischsort varArr
            
            For i = 0 To UBound(varTable(0))
                strArr(i) = strTemp(Split(varArr(i), ",")(1))
            Next
            strOutput = Join(strArr, strDelimiter)
        Else
            
            If bolSort Then Me.Mischsort strTemp
            strOutput = Join(strTemp, strDelimiter)
        End If
        
        If Len(strOutput) > 1 And right(strOutput, 1) = strDelimiter Then strOutput = Left(strOutput, Len(strOutput) - 1)
        fnCountryValueList = strOutput
    Else
        fnCountryValueList = ""
    End If
End Function

'---------------------------------------------------------------------------------------
' Procedure   : fnListSeparator
' Date        : 10.02.2013
' Purpose     : Gets the system's list separator which is used in value lists (i.e. comboboxes/listboxes)
' Parameters  : lngLocaleType : SYSTEM/USER/INVARIANT settings (or any other wanted language)
' Returns     : var - Variant
'---------------------------------------------------------------------------------------
'
Public Function fnListSeparator(Optional ByVal lngLocaleType As EnmCCRegionalsType = A_LOCALE_USER_DEFAULT)
    fnListSeparator = Me.fnLocaleInfo(LOCALE_SLIST, lngLocaleType)
End Function

Private Function fnParseDate(ByVal strValue As String) As String
    Dim i As Long, p As Long, k As Long
    Dim bytMonth As Byte
    Dim intYear As Integer
    Dim bytDay As Byte
    Dim strMonthNames() As String
    Dim strChar As String * 1
    Dim strYear As String
    Dim strResult As String
    
    For i = UBound(prv_varDayReplace) To 0 Step -1
        If InStr(strValue, prv_varDayReplace(i)) > 0 Then
            bytDay = i + 1
            strValue = Replace(strValue, prv_varDayReplace(i), "")
        End If
    Next
    
    For p = 0 To 1  ' Long/Short
        For i = 0 To UBound(prv_varAllLocales)
            strMonthNames = Me.fnMonthNames(p, prv_varAllLocales(i))
            For k = 1 To 12
                If strMonthNames(k) <> "" And Len(strMonthNames(k)) > 1 And (InStr(strValue, strMonthNames(k) & " ") > 0 Or InStr(strValue, " " & strMonthNames(k)) > 0) Then
                    bytMonth = k
                    strValue = Replace(strValue, strMonthNames(k), "")
                    Exit For
                End If
            Next
            If bytMonth > 0 Then Exit For
        Next
        If bytMonth > 0 Then Exit For
    Next
        
    If bytMonth > 0 And bytDay > 0 Then
        For i = 1 To Len(strValue)
            strChar = Mid(strValue, i, 1)
            If Asc(strChar) >= 48 And Asc(strChar) <= 57 Then
                strYear = strYear & strChar
            End If
            If (Asc(strChar) < 48 Or Asc(strChar) > 57) And strYear <> "" Then Exit For
        Next
    End If
    If strYear <> "" Then
        intYear = Val(strYear)
        strValue = Replace(strValue, intYear, "")
    End If
    
    Select Case Me.fnLocaleInfo(LOCALE_IDATE, A_LOCALE_USER_DEFAULT)   ' 0 = M-D-Y, 1 = D-M-Y,2 = Y-M-D
        Case 0
            strResult = bytMonth & "/" & bytDay & "/" & intYear
        Case 1
            strResult = bytDay & "/" & bytMonth & "/" & intYear
        Case 2
            strResult = intYear & "/" & bytMonth & "/" & bytDay
    End Select
    fnParseDate = strResult & " " & Trim(strValue)
End Function

'---------------------------------------------------------------------------------------
' Property     : fnValidateDateAndTime
' Date         : 23.01.2013
' Purpose      : Validates the value if it is a valid date string. First tries to use CDate to convert, if that fails,
'                tries to parse the date/time string. If the result of the date string cannot be found out from the values,
'                then the order dd-mm-yy is used to convert
'                (Access ALWAYS formats the "date and time" string as "Date Time AM/PM/Nothing", independent of the system settings!
'                 For example: Windows Setting is "th|mm|ss" with "AM_" and "PM_" then Access formats it as "8|21|08 AM_" where Windows
'                 would format it as "A8|21|08".)
' Requirements : - if the value string contains a time string then it must be located after the date part and separated with one space
'                - the date string can also contain text month names if the chosen locale type contains MMM or MMMM in the date format.
'                - the date string must be formatted as short date, long date formats will not be checked
'                - the date format separator must not be space
' Parameters   : strValue: Value to test
' Returns      : bol - Boolean: False, if it is no valid date string
'---------------------------------------------------------------------------------------
'
Public Function fnValidateDateAndTime(ByVal strValue As String, _
                                      Optional ByVal intCompareMethod As VbCompareMethod = vbTextCompare) As Boolean
    Dim strMonths() As String
    Dim strMonthsUser() As String
    Dim strValueInitial As String
    Dim strValues() As String
    Dim strValues2() As String
    Dim strTestValue As String
    Dim strTestValue2 As String
    Dim i As Long, k As Long, n As Long, p As Long, q As Long
    Dim bolDateError As Boolean
    
    Class_Initialize
    
    prv_datDate = 0
    fnValidateDateAndTime = False
    
    If IsDate(strValue) Then
        prv_datDate = CDate(strValue)
        fnValidateDateAndTime = True
        GoTo fnValidateDate_Exit    '--------------------->
    End If
    
    For i = 0 To UBound(prv_varSeparators)
        strValue = Replace(strValue, prv_varSeparators(i) & " ", prv_varSeparators(i))
    Next
    
    Do While InStr(strValue, "  ") > 0
        strValue = Replace(strValue, "  ", " ")                 ' Replace any double spaces with single spaces
    Loop
    
    strValueInitial = strValue
    
    strValues = Split(strValue, " ")                            ' split date and time part (must always be separated with space)
    If UBound(strValues) > 2 Then
        strValue = fnParseDate(strValue)
        strValues = Split(strValue, " ")
    End If
    
    For q = 1 To 3
        bolDateError = True
        strValue = strValueInitial
        If q = 2 Then
            ' This test only applies in the second loop
            If UBound(strValues) > 1 Then                           ' more than 1 space used in date/time string
                strValue = Replace(strValue, " ", "", , 2)
                strValues = Split(strValue, " ")                    ' split date and time part (must always be separated with space)
            End If
        End If
        
        If q = 3 Then
            strValue = fnParseDate(strValue)
            strValues = Split(strValue, " ")                        ' split date and time part (must always be separated with space)
        End If
        
        For p = 0 To UBound(strValues)
            For i = 0 To UBound(prv_varSeparators)
                strTestValue = Replace(Replace(strValues(p), prv_varSeparators(i), "/", , , intCompareMethod), " ", "") ' Replace separator with standard separator for CDate test
                If IsDate(strTestValue) Then
                    prv_datDate = CDate(strTestValue)
                    strValues(p) = strTestValue
                    bolDateError = False
                    Exit For
                End If
                strMonthsUser = prv_varMonthNames(1)  ' User locale long month names to convert to (see Class_Initialize)
                For n = 3 To UBound(prv_varMonthNames) Step 2  ' English/German long month names to convert from
                    strMonths = prv_varMonthNames(n)
                    For k = 1 To 12
                        strTestValue2 = Replace(strTestValue, strMonths(k), strMonthsUser(k), , , intCompareMethod)
                        If IsDate(strTestValue2) Then
                            prv_datDate = CDate(strTestValue2)
                            strValues(p) = strTestValue2
                            bolDateError = False
                            Exit For
                        End If
                    Next
                Next
                strMonthsUser = prv_varMonthNames(0)  ' User locale short month names to convert to (see Class_Initialize)
                For n = 2 To UBound(prv_varMonthNames) Step 2  ' English/German short month names to convert from
                    strMonths = prv_varMonthNames(n)
                    For k = 1 To 12
                        strTestValue2 = Replace(strTestValue2, strMonths(k), strMonthsUser(k), , , intCompareMethod)
                        If IsDate(strTestValue2) Then
                            prv_datDate = CDate(strTestValue2)
                            strValues(p) = strTestValue2
                            bolDateError = False
                            Exit For
                        End If
                    Next
                Next
            Next
            If bolDateError = False Then Exit For
        Next
        
        If bolDateError = False Then
            strValue = strValues(p)
            k = 0
            For i = 0 To UBound(strValues)
                If i <> p Then
                    ReDim Preserve strValues2(k)
                    strValues2(k) = strValues(i)
                    k = k + 1
                End If
            Next
            ' Try to assemble the date and time string and convert with CDate
            strValue = strValue & " " & Join(strValues2, " ")
            If IsDate(strValue) Then
                prv_datDate = CDate(strValue)
                fnValidateDateAndTime = True
            Else
                ' now test the time string with different separators
                For p = 0 To UBound(strValues2)
                    ' First try to replace AM/PM strings by standard "AM" / "PM"
                    For i = 2 To UBound(prv_varAMPMs) Step 2
                        strTestValue = strValues2(p)
                        If prv_varAMPMs(i) <> "" Then
                            strTestValue = Replace(strValues2(p), prv_varAMPMs(i), " AM ", , , intCompareMethod)
                            If strTestValue <> strValues2(p) Then Exit For
                        End If
                        If prv_varAMPMs(i + 1) <> "" Then
                            strTestValue = Replace(strValues2(p), prv_varAMPMs(i + 1), " PM ", , , intCompareMethod)
                            If strTestValue <> strValues2(p) Then Exit For
                        End If
                    Next
                    If strTestValue <> strValues2(p) Then
                        strValues2(p) = strTestValue
                        Exit For
                    End If
                Next
                strTestValue2 = Join(strValues2, " ")
                For i = 0 To UBound(prv_varSeparators)
                    If prv_varSeparators(i) <> ":" Then
                        strTestValue = Replace(strTestValue2, prv_varSeparators(i), ":", , , intCompareMethod) ' Replace separator with standard separator for CDate test
                        If IsDate(strTestValue) Then
                            prv_datDate = prv_datDate + CDate(strTestValue)
                            fnValidateDateAndTime = True
                            Exit For
                        End If
                    End If
                Next
            End If
        End If
        If bolDateError = False Then Exit For
    Next

    
fnValidateDate_Exit:
End Function

'---------------------------------------------------------------------------------------
' Property     : fnConvertToSQLDate
' Date         : 24.01.2013
' Purpose      : Converts an input value into SQL usable date standard string using the Windows Regional Settings,
'                performs automatically "fnValidateDateAndTime".
' Parameters   : strValue: Date string formatted in the desired locale form, any other value type will be tried to be converted to date/string
'                intCompareMethod: A VB compare method. vbBinaryCompare is exact,
'                                  vbTextCompare also allows different upper/lowercase etc., Default is vbTextCompare
'                intTarget: Access or SQL Server output format
'                bolWithSingleQuotes: For SQL Server only: Surrounds the output string with single quotes
'                bolNULLOnError: See below
' Returns      : var - Variant: If "bolNULLOnError" is True then it returns the string "NULL", else "ERROR" as string
'---------------------------------------------------------------------------------------
'
Public Function fnConvertToSQLDate(ByVal strValue As Variant, _
                                   Optional ByVal intCompareMethod As VbCompareMethod = vbTextCompare, _
                                   Optional ByVal intDateAndTime As EnmCCRegionalsDateTime = enmCCReg_DateAndTime, _
                                   Optional ByVal intTarget As EnmCCRegionalsTarget = enmCCRegTarget_Access, _
                                   Optional ByVal bolWithSinqleQuotes As Boolean = True, _
                                   Optional ByVal bolNULLOnError As Boolean = True) As String
    Dim bolError As Boolean
    
    bolError = False
    Select Case VarType(strValue)
        Case vbEmpty, vbNull
            fnConvertToSQLDate = "NULL"
            GoTo fnConvertToSQLDate_Exit  ' --------------------->
        Case vbInteger, vbLong, vbSingle, vbDouble, vbCurrency, vbDecimal, vbByte
            On Error Resume Next
            ' Max value before decimal point is 2958465 (12/31/9999) and 99999 after decimal point (23:59:59)
            ' Access can use a min value of -657434 (01/01/0100), SQL Server (2005) only 0
            ' 0 = 12/30/1899 in Access and 01/01/1900 in SQL Server (2005)
            strValue = CDate(strValue)
            If Err.Number = 0 Then
                strValue = Format(strValue, cFORMAT_StandardDateTime)
            Else
                bolError = True
            End If
            On Error GoTo 0
        Case vbDate
            strValue = Format(strValue, cFORMAT_StandardDateTime)
        Case vbString
        Case Else
            bolError = True
    End Select
    
    If bolError = False Then
        If fnValidateDateAndTime(CStr(Nz(strValue))) Then
            Select Case intDateAndTime
                Case enmCCReg_DateOnly
                    fnConvertToSQLDate = Me.SQLDateOnly(intTarget, bolWithSinqleQuotes)
                Case enmCCReg_TimeOnly
                    fnConvertToSQLDate = Me.SQLTimeOnly(intTarget, bolWithSinqleQuotes)
                Case enmCCReg_DateAndTime
                    fnConvertToSQLDate = Me.SQLDateAndTime(intTarget, bolWithSinqleQuotes)
            End Select
        Else
            bolError = True
        End If
    End If
    
    If bolError Then
        If bolNULLOnError Then
            fnConvertToSQLDate = "NULL"
        Else
            fnConvertToSQLDate = "ERROR"
        End If
    End If
    
fnConvertToSQLDate_Exit:
End Function

'---------------------------------------------------------------------------------------
' Property     : fnLeapYear
' Date         : 23.01.2013
' Purpose      : Calculates if the specified year is a leap year
' Returns      : bol - Boolean
'---------------------------------------------------------------------------------------
'
Public Function fnLeapYear(intYear As Integer) As Boolean
    fnLeapYear = False
    If intYear Mod 400 = 0 Then
        fnLeapYear = True
    Else
        If intYear Mod 4 = 0 And intYear Mod 100 <> 0 Then fnLeapYear = True
    End If
End Function

'---------------------------------------------------------------------------------------
' Procedure   : Mischsort
'              (Original algorithm author: Horst Armin Kosog
'               from magazine "64er" edition 9/1985, pages 126/127)
'               Details see: http://suche.administrator.de/wissen/arrays-in-vbscript-sehr-schnell-sortieren-mischsort-von-1985-74756.html
' Author      : Christian Coppes (translated algorithm for VBA)
' Date        : 29.11.2007
' Last Change : 17.02.2013
' Purpose     : Extremely fast sort algorithm, faster than QuickSort or BubbleSort
' Parameters  : varArr: An array to be sorted
' Returns     :  -
'---------------------------------------------------------------------------------------
'
Public Sub Mischsort(ByRef varArr As Variant)
    Dim A, intTeilfeldAnzahl, X
    Dim intLaufendeFeldlaenge, intMischzaehler, intTeilfeldZaehler
    Dim A1, A2, E1, E2, ET, Y
    Dim ZF, LF, MZ, FZ, NextX
    Dim i As Long
    Dim strTempArr() As Variant
    Dim strArr() As Variant
    
    ' Shift the array from 1 to x because algorithm only works with arrays beginning with index 1
    If LBound(varArr) = 0 Then
        ReDim strArr(1 To UBound(varArr) + 1)
        For i = 0 To UBound(varArr)
            strArr(1 + i) = varArr(i)
        Next
    End If
    A = UBound(strArr)
                                
    ZF = Int(A / 2 + 0.5)
    ReDim strTempArr(ZF)
                                
    For X = 2 To A Step 2
        If strArr(X - 1) > strArr(X) Then
            strTempArr(0) = strArr(X - 1)
            strArr(X - 1) = strArr(X)
            strArr(X) = strTempArr(0)
        End If
    Next 'X
    If A < 3 Then Exit Sub
    LF = 1
    For MZ = 1 To Int(Log(A - 1) / Log(2))
        ZF = Int(ZF / 2 + 0.5)
        LF = 2 * LF
        For FZ = 1 To ZF
            A1 = 1 + 2 * LF * (FZ - 1)
            A2 = A1 + LF
            E1 = A2 - 1
            E2 = E1 + LF
            If E2 <= A Then
                ET = LF
            Else
                If Not (A < A2) Then
                    E2 = A
                    ET = A + 1 - A2
                End If ' Not (A < A2)
            End If ' E2 <= A
                                
            If (E2 <= A) Or (Not (A < A2)) Then
                For X = 1 To ET
                    strTempArr(X) = strArr(X + E1)
                Next
                For X = E2 To A1 Step -1
                    NextX = 0
                    If strArr(E1) < strTempArr(ET) Then
                        NextX = 1
                        strArr(X) = strTempArr(ET)
                        ET = ET - 1
                        If Not (ET >= 1) Then
                            X = A1
                        End If 'Not (ET >= 1)
                    End If 'strArr(E1) < strTempArr(ET)
                                
                    If NextX = 0 Then
                        strArr(X) = strArr(E1)
                        E1 = E1 - 1
                        If Not (E1 >= A1) Then
                            For Y = X - 1 To A1 Step -1
                                strArr(Y) = strTempArr(ET)
                                ET = ET - 1
                            Next 'Y
                            X = A1
                        End If 'Not (E1 >= A1)
                    End If 'NextX = 0
                Next X
            End If '(E2<=A) Or (Not(A<A2))
        Next FZ
    Next MZ
    
MischSort_Exit:
    ' Shift the array back to 0 index
    If LBound(varArr) = 0 Then
        For i = 0 To UBound(varArr)
            varArr(i) = strArr(1 + i)
        Next
    End If
End Sub


'#######################################################################################
'---------------------------------------------------------------------------------------
'------------------------------ Constructor/Destructor ---------------------------------
'---------------------------------------------------------------------------------------
'#######################################################################################

Private Sub Class_Initialize()
    Dim i As Long
    Dim lngReturn As Long
    
    ' Number of days per month
    prv_varDays = Array(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)
    ' Separators which will be tested
    prv_varSeparators = Array(".", "|", "_", "\", "#", "*", "+", "~")
    ReDim Preserve prv_varSeparators(UBound(prv_varSeparators) + 1)
    ' Add current user locale separator to the separator test array
    prv_varSeparators(UBound(prv_varSeparators)) = Me.fnLocaleInfo(LOCALE_SDATE, A_LOCALE_USER_DEFAULT)
    
    ' Default locales which will be tested in fnValidateDateAndTime, add more like you want
    ' (only the first one is needed and one additional)
    prv_varTestLocales = Array(A_LOCALE_USER_DEFAULT, LOCALE_English_UnitedStates, LOCALE_German_Germany, _
                               LOCALE_Italian_Italy, LOCALE_French_France, LOCALE_Swedish_Sweden, _
                               LOCALE_Dutch_Netherlands, LOCALE_Spanish_Spain_Modern, LOCALE_Slovak, _
                               LOCALE_Polish, LOCALE_Portuguese_Portugal)
    ReDim prv_varAMPMs(((UBound(prv_varTestLocales) + 1) * 2) - 1)
    ReDim prv_varMonthNames(((UBound(prv_varTestLocales) + 1) * 2) - 1)
    
    ' USER_DEFAULT = Month names to convert a date string into
    For i = 0 To UBound(prv_varTestLocales)
        prv_varMonthNames(i * 2) = Me.fnMonthNames(LOCALENAME_short, prv_varTestLocales(i))
        prv_varMonthNames(i * 2 + 1) = Me.fnMonthNames(LOCALENAME_Long, prv_varTestLocales(i))
        prv_varAMPMs(i * 2) = Me.fnLocaleInfo(LOCALE_S1159, prv_varTestLocales(i))
        prv_varAMPMs(i * 2 + 1) = Me.fnLocaleInfo(LOCALE_S2359, prv_varTestLocales(i))
    Next
    
    prv_varDayReplace = Array("1st", "2nd", "3rd", "4th", "5th", "6th", "7th", "8th", "9th", "10th", "11th", "12th", "13th", "14th", "15th", "16th", _
                              "17th", "18th", "19th", "20th", "21st", "22nd", "23rd", "24th", "25th", "26th", "27th", "28th", "29th", "30th", "31st")
    
    prv_varAllCountries = Array(LOCALE_Albanian, LOCALE_Arabic_UnitedArabEmirates, LOCALE_Arabic_Bahrain, LOCALE_Arabic_Algeria, LOCALE_Arabic_Egypt, LOCALE_Arabic_Iraq, LOCALE_Arabic_Jordan, _
                                LOCALE_Arabic_Kuwait, LOCALE_Arabic_Lebanon, LOCALE_Arabic_Libya, LOCALE_Arabic_Morocco, LOCALE_Arabic_Oman, LOCALE_Arabic_Qatar, LOCALE_Arabic_SaudiArabia, LOCALE_Arabic_Syria, _
                                LOCALE_Arabic_Tunisia, LOCALE_Arabic_Yemen, LOCALE_Armenian, LOCALE_Azeri_Latin, LOCALE_Belarusian, LOCALE_Bulgarian, _
                                LOCALE_Chinese_China, LOCALE_Chinese_HongKongSAR, LOCALE_Chinese_MacauSAR, LOCALE_Chinese_Taiwan, LOCALE_Croatian, LOCALE_Czech, LOCALE_Danish, LOCALE_Dutch_Netherlands, _
                                LOCALE_Dutch_Belgium, LOCALE_English_Australia, LOCALE_English_Belize, LOCALE_English_Canada, LOCALE_English_Caribbean, LOCALE_English_India, LOCALE_English_Jamaica, _
                                LOCALE_English_NewZealand, LOCALE_English_Phillippines, LOCALE_English_Singapore, LOCALE_English_SouthernAfrica, LOCALE_English_Trinidad, LOCALE_English_GreatBritain, _
                                LOCALE_English_UnitedStates, LOCALE_English_Zimbabwe, LOCALE_Estonian, LOCALE_Farsi, LOCALE_Finnish, LOCALE_Faroese, LOCALE_French_France, _
                                LOCALE_French_Luxembourg, LOCALE_Irish_Ireland, LOCALE_German_Germany, LOCALE_German_Austria, LOCALE_German_Liechtenstein, _
                                LOCALE_German_Switzerland, LOCALE_Greek, LOCALE_Hebrew, LOCALE_Hungarian, LOCALE_Icelandic, LOCALE_Indonesian, LOCALE_Italian_Italy, LOCALE_Japanese, LOCALE_Korean, _
                                LOCALE_Latvian, LOCALE_Lithuanian, LOCALE_FYRO_Macedonia, LOCALE_Malay_Malaysia, LOCALE_Malay_Brunei, LOCALE_Maltese, LOCALE_Norwegian_Bokmal, LOCALE_Polish, _
                                LOCALE_Portuguese_Portugal, LOCALE_Portuguese_Brazil, LOCALE_Romanian_Romania, LOCALE_Russian, _
                                LOCALE_Serbian_Latin, LOCALE_Slovenian, LOCALE_Slovak, LOCALE_Spanish_Spain_Modern, LOCALE_Spanish_Argentina, _
                                LOCALE_Spanish_Bolivia, LOCALE_Spanish_Chile, LOCALE_Spanish_Colombia, LOCALE_Spanish_CostaRica, LOCALE_Spanish_DominicanRepublic, LOCALE_Spanish_Ecuador, LOCALE_Spanish_Guatemala, LOCALE_Spanish_Honduras, _
                                LOCALE_Spanish_Mexico, LOCALE_Spanish_Nicaragua, LOCALE_Spanish_Panama, LOCALE_Spanish_Peru, LOCALE_Spanish_PuertoRico, LOCALE_Spanish_Paraguay, LOCALE_Spanish_ElSalvador, LOCALE_Spanish_Uruguay, LOCALE_Spanish_Venezuela, _
                                LOCALE_Swahili, LOCALE_Swedish_Sweden, LOCALE_Tamil, LOCALE_Thai, LOCALE_Turkish, LOCALE_Ukrainian, LOCALE_Urdu, LOCALE_Uzbek_Latin, _
                                LOCALE_Vietnamese)
    
    prv_varAllLocales = Array(LOCALE_Afrikaans, LOCALE_Albanian, LOCALE_Arabic_UnitedArabEmirates, LOCALE_Arabic_Bahrain, LOCALE_Arabic_Algeria, LOCALE_Arabic_Egypt, LOCALE_Arabic_Iraq, LOCALE_Arabic_Jordan, _
                              LOCALE_Arabic_Kuwait, LOCALE_Arabic_Lebanon, LOCALE_Arabic_Libya, LOCALE_Arabic_Morocco, LOCALE_Arabic_Oman, LOCALE_Arabic_Qatar, LOCALE_Arabic_SaudiArabia, LOCALE_Arabic_Syria, _
                              LOCALE_Arabic_Tunisia, LOCALE_Arabic_Yemen, LOCALE_Armenian, LOCALE_Azeri_Latin, LOCALE_Azeri_Cyrillic, LOCALE_Basque, LOCALE_Belarusian, LOCALE_Bulgarian, LOCALE_Catalan, _
                              LOCALE_Chinese_China, LOCALE_Chinese_HongKongSAR, LOCALE_Chinese_MacauSAR, LOCALE_Chinese_Singapore, LOCALE_Chinese_Taiwan, LOCALE_Croatian, LOCALE_Czech, LOCALE_Danish, LOCALE_Dutch_Netherlands, _
                              LOCALE_Dutch_Belgium, LOCALE_English_Australia, LOCALE_English_Belize, LOCALE_English_Canada, LOCALE_English_Caribbean, LOCALE_English_India, LOCALE_English_Ireland, LOCALE_English_Jamaica, _
                              LOCALE_English_Malaysia, LOCALE_English_NewZealand, LOCALE_English_Phillippines, LOCALE_English_Singapore, LOCALE_English_SouthernAfrica, LOCALE_English_Trinidad, LOCALE_English_GreatBritain, _
                              LOCALE_English_UnitedStates, LOCALE_English_Zimbabwe, LOCALE_Estonian, LOCALE_Farsi, LOCALE_Finnish, LOCALE_Faroese, LOCALE_French_France, LOCALE_French_Belgium, LOCALE_French_Canada, _
                              LOCALE_French_Luxembourg, LOCALE_French_Switzerland, LOCALE_Irish_Ireland, LOCALE_ScottishGaelic_UnitedKingdom, LOCALE_German_Germany, LOCALE_German_Austria, LOCALE_German_Liechtenstein, LOCALE_German_Luxembourg, _
                              LOCALE_German_Switzerland, LOCALE_Greek, LOCALE_Hebrew, LOCALE_Hindi, LOCALE_Hungarian, LOCALE_Icelandic, LOCALE_Indonesian, LOCALE_Italian_Italy, LOCALE_Italian_Switzerland, LOCALE_Japanese, LOCALE_Korean, _
                              LOCALE_Latvian, LOCALE_Lithuanian, LOCALE_FYRO_Macedonia, LOCALE_Malay_Malaysia, LOCALE_Malay_Brunei, LOCALE_Maltese, LOCALE_Marathi, LOCALE_Norwegian_Bokmal, LOCALE_Norwegian_Nynorsk, LOCALE_Polish, _
                              LOCALE_Portuguese_Portugal, LOCALE_Portuguese_Brazil, LOCALE_Raeto_Romance, LOCALE_Romanian_Romania, LOCALE_Romanian_RepublicOfMoldova, LOCALE_Russian, LOCALE_Russian_RepublicOfMoldova, LOCALE_Sanskrit, _
                              LOCALE_Serbian_Cyrillic, LOCALE_Serbian_Latin, LOCALE_Setsuana, LOCALE_Slovenian, LOCALE_Slovak, LOCALE_Sorbian, LOCALE_Spanish_Spain_Modern, LOCALE_Spanish_Spain_Traditional, LOCALE_Spanish_Argentina, _
                              LOCALE_Spanish_Bolivia, LOCALE_Spanish_Chile, LOCALE_Spanish_Colombia, LOCALE_Spanish_CostaRica, LOCALE_Spanish_DominicanRepublic, LOCALE_Spanish_Ecuador, LOCALE_Spanish_Guatemala, LOCALE_Spanish_Honduras, _
                              LOCALE_Spanish_Mexico, LOCALE_Spanish_Nicaragua, LOCALE_Spanish_Panama, LOCALE_Spanish_Peru, LOCALE_Spanish_PuertoRico, LOCALE_Spanish_Paraguay, LOCALE_Spanish_ElSalvador, LOCALE_Spanish_Uruguay, LOCALE_Spanish_Venezuela, _
                              LOCALE_SouthernSotho, LOCALE_Swahili, LOCALE_Swedish_Sweden, LOCALE_Swedish_Finland, LOCALE_Tamil, LOCALE_Tatar, LOCALE_Thai, LOCALE_Turkish, LOCALE_Tsonga, LOCALE_Ukrainian, LOCALE_Urdu, LOCALE_Uzbek_Cyrillic, LOCALE_Uzbek_Latin, _
                              LOCALE_Vietnamese, LOCALE_Xhosa, LOCALE_Yiddish, LOCALE_Zulu)
    prv_varAllLocaleNames = Array("Afrikaans", "Albanian", "Arabic_UnitedArabEmirates", "Arabic_Bahrain", "Arabic_Algeria", "Arabic_Egypt", "Arabic_Iraq", "Arabic_Jordan", _
                                  "Arabic_Kuwait", "Arabic_Lebanon", "Arabic_Libya", "Arabic_Morocco", "Arabic_Oman", "Arabic_Qatar", "Arabic_SaudiArabia", "Arabic_Syria", _
                                  "Arabic_Tunisia", "Arabic_Yemen", "Armenian", "Azeri_Latin", "Azeri_Cyrillic", "Basque", "Belarusian", "Bulgarian", "Catalan", _
                                  "Chinese_China", "Chinese_HongKongSAR", "Chinese_MacauSAR", "Chinese_Singapore", "Chinese_Taiwan", "Croatian", "Czech", "Danish", "Dutch_Netherlands", _
                                  "Dutch_Belgium", "English_Australia", "English_Belize", "English_Canada", "English_Caribbean", "English_India", "English_Ireland", "English_Jamaica", _
                                  "English_Malaysia", "English_NewZealand", "English_Phillippines", "English_Singapore", "English_SouthernAfrica", "English_Trinidad", "English_GreatBritain", _
                                  "English_UnitedStates", "English_Zimbabwe", "Estonian", "Farsi", "Finnish", "Faroese", "French_France", "French_Belgium", "French_Canada", _
                                  "French_Luxembourg", "French_Switzerland", "Irish_Ireland", "ScottishGaelic_UnitedKingdom", "German_Germany", "German_Austria", "German_Liechtenstein", "German_Luxembourg", _
                                  "German_Switzerland", "Greek", "Hebrew", "Hindi", "Hungarian", "Icelandic", "Indonesian", "Italian_Italy", "Italian_Switzerland", "Japanese", "Korean", _
                                  "Latvian", "Lithuanian", "FYRO_Macedonia", "Malay_Malaysia", "Malay_Brunei", "Maltese", "Marathi", "Norwegian_Bokmal", "Norwegian_Nynorsk", "Polish", _
                                  "Portuguese_Portugal", "Portuguese_Brazil", "Raeto_Romance", "Romanian_Romania", "Romanian_RepublicOfMoldova", "Russian", "Russian_RepublicOfMoldova", "Sanskrit", _
                                  "Serbian_Cyrillic", "Serbian_Latin", "Setsuana", "Slovenian", "Slovak", "Sorbian", "Spanish_Spain_Modern", "Spanish_Spain_Traditional", "Spanish_Argentina", _
                                  "Spanish_Bolivia", "Spanish_Chile", "Spanish_Colombia", "Spanish_CostaRica", "Spanish_DominicanRepublic", "Spanish_Ecuador", "Spanish_Guatemala", "Spanish_Honduras", _
                                  "Spanish_Mexico", "Spanish_Nicaragua", "Spanish_Panama", "Spanish_Peru", "Spanish_PuertoRico", "Spanish_Paraguay", "Spanish_ElSalvador", "Spanish_Uruguay", "Spanish_Venezuela", _
                                  "SouthernSotho", "Swahili", "Swedish_Sweden", "Swedish_Finland", "Tamil", "Tatar", "Thai", "Turkish", "Tsonga", "Ukrainian", "Urdu", "Uzbek_Cyrillic", "Uzbek_Latin", _
                                  "Vietnamese", "Xhosa", "Yiddish", "Zulu")

    ' enter additional paper sizes if known. Must be at the position of the Windows Locale value "LOCALE_IPAPERSIZE".
    prv_varPaperSizes = Array("", "US Letter", "", "", "", "US Legal", "", "", "A3", "A4", "", "", "", "", "", "", "", "", "", "", "", "", "", "")

    ' calculate the size of the type structure
    prv_typWinVersion.lngWinVersionInfoSize = Len(prv_typWinVersion)
    ' Fill the structure with Windows version infos
    lngReturn = GetVersionEx(prv_typWinVersion)
    Select Case prv_typWinVersion.lngMajorVersion
        Case Is <= 5
            ' everything lower than Windows XP will be handled as Windows XP
            prv_lngWinVersion = Version_WinXP
        Case 6
            prv_lngWinVersion = Version_WinVista
        Case 7
            prv_lngWinVersion = Version_Win7
    End Select
End Sub

